import 'package:flutter/material.dart';

import 'package:flutter_custom_toolbox/flutter_toolbox.dart';

class PageParameters extends StatelessWidget {
  const PageParameters({
    super.key,
    required this.appConfig,
    required this.canBeResumed,
  });

  final ApplicationConfigDefinition appConfig;
  final bool canBeResumed;

  final double separatorHeight = 8.0;

  @override
  Widget build(BuildContext context) {
    final List<Widget> lines = [];

    // Activity settings (top)
    for (ApplicationSettingsParameter parameter in appConfig.activitySettings) {
      if (parameter.displayedOnTop) {
        for (var itemsLine in buildParametersLines(
          parameter: parameter,
          isEnabled: !canBeResumed,
        )) {
          lines.add(Row(
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            children: itemsLine,
          ));
        }

        lines.add(SizedBox(height: separatorHeight));
      }
    }

    lines.add(Expanded(
      child: SizedBox(height: separatorHeight),
    ));

    if (canBeResumed) {
      // Resume activity
      lines.add(AspectRatio(
        aspectRatio: 3,
        child: ActivityButtonResumeSaved(
          onPressed: () {
            appConfig.resumeActivity(context);
          },
        ),
      ));
      // Delete saved activity
      lines.add(SizedBox.square(
        dimension: MediaQuery.of(context).size.width / 5,
        child: ActivityButtonDeleteSaved(
          onPressed: () {
            appConfig.deleteCurrentActivity(context);
          },
        ),
      ));
    } else {
      // Start new activity
      lines.add(
        AspectRatio(
          aspectRatio: 3,
          child: ActivityButtonStartNew(
            onPressed: () {
              appConfig.startNewActivity(context);
            },
          ),
        ),
      );
    }

    lines.add(SizedBox(height: separatorHeight));

    // Activity settings (bottom)
    for (ApplicationSettingsParameter parameter in appConfig.activitySettings) {
      if (!parameter.displayedOnTop) {
        for (var itemsLine in buildParametersLines(
          parameter: parameter,
          isEnabled: !canBeResumed,
        )) {
          lines.add(Row(
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            children: itemsLine,
          ));
        }

        lines.add(SizedBox(height: separatorHeight));
      }
    }

    return Column(
      children: lines,
    );
  }

  List<List<Widget>> buildParametersLines({
    required ApplicationSettingsParameter parameter,
    required bool isEnabled,
  }) {
    final List<ApplicationSettingsParameterItemValue> items = parameter.values;
    final List<Widget> parameterButtons = [];

    if (items.length <= 1) {
      return [];
    }

    final int itemsPerLine =
        parameter.itemsPerLine != 0 ? parameter.itemsPerLine : items.length;

    // Build linear list
    for (ApplicationSettingsParameterItemValue item in items) {
      final Widget parameterButton = BlocBuilder<ActivitySettingsCubit, ActivitySettingsState>(
        builder: (BuildContext context, ActivitySettingsState activitySettingsState) {
          final ActivitySettingsCubit activitySettingsCubit =
              BlocProvider.of<ActivitySettingsCubit>(context);

          final double displayWidth = MediaQuery.of(context).size.width;
          final double itemWidth = displayWidth / itemsPerLine - 4;

          return SizedBox.square(
            dimension: itemWidth,
            child: parameter.buildParameterItem(
              context: context,
              parameter: parameter,
              itemValue: item,
              size: itemWidth,
              onPressed: isEnabled
                  ? () {
                      activitySettingsCubit.set(parameter.code, item.value);
                    }
                  : null,
            ),
          );
        },
      );

      parameterButtons.add(parameterButton);
    }

    // Split buttons in lines according to items count limit per line
    final List<List<Widget>> lines = [];

    List<Widget> line = [];
    for (var parameterButton in parameterButtons) {
      line.add(parameterButton);
      if (line.length == itemsPerLine) {
        lines.add(line);
        line = [];
      }
    }
    if (line.length != itemsPerLine) {
      lines.add(line);
    }

    return lines;
  }
}
